<?php
namespace DAO;

use Our\ApiConst;
use Our\DbNameConst;
use Our\ImageConst;

/**
 * 店铺数据层
 * @date 2018-5-14
 * @author csw
 */
class StoreModel extends \DAO\AbstractModel
{


    protected function init()
    {

    }

    /**
     * 表名
     *
     * @var string
     */
    protected $_tableName = 'han_store';

    /**
     * 主键
     *
     * @var string
     */
    protected $_primaryKey = 'store_id';



    /**
     * 获取店铺信息
     * @param $where
     * @return mixed
     */

    public function getInfo($where)
    {
        $this->setDb();
        $field = $this->_getColumns();
        $store = $this->db->select($field)->from($this->_tableName)->where($where)->fetchOne();
        $store['store_label'] = $this->getStoreLabelSrc($store['store_label']);
        return $store;
    }

    /**
     * 获取店铺评分
     * @param $StoreId
     * @return mixed
     */
    public function getGrade($StoreId, $storeInfo = null)
    {
        $statisticsDb6Redis = \Redis\Db6\StatisticsRedisModel::getInstance();
        $grade = $statisticsDb6Redis->tableHGet('store', $StoreId);
        if ($grade === false) {
            if (!$storeInfo) {
                $storeScoreDAO = \DAO\StoreScoreStatisticsModel::getInstance();
                $storeInfo = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreScoreStatisticsRedisModel::getInstance(), array(&$storeScoreDAO, 'getOne'), array('*',array('store_id'=>$StoreId)),\Our\ApiConst::oneDaySecond);
                if($storeInfo === false){
                    $storeInfo['evaluation_score'] = 0;
                    $storeInfo['evaluation_number'] = 0;
                }
            }
            $grade = array();
            $grade['evaluation_score'] = $storeInfo['evaluation_score'];
            $grade['evaluation_number'] = $storeInfo['evaluation_number'];
            $statisticsDb6Redis->tableHSet('store', $StoreId, serialize($grade));
            return $grade;
        }
        return unserialize($grade);
    }

    public function getInfoById($id, $field = '*', $isField = false)
    {
        $this->setDb($this->dbName);
        $where[$this->_primaryKey] = $id;
        $store = $this->db->from($this->_tableName)->select($field)->where($where)->fetchOne();
        if ($isField) {
            return $store[$isField];
        } else {
            return $store;
        }
    }

    public function getInfoByMemberId($memberId, $field = '*', $isField = false, $db = DbNameConst::salveDBConnectName)
    {
        $this->setDb($db);
        $where['member_id'] = $memberId;
        $store = $this->db->from($this->_tableName)->select($field)->where($where)->fetchOne();
        if ($isField) {
            return $store[$isField];
        } else {
            return $store;
        }
    }

    /**
     * 店铺头像
     *
     * @param $storeLabel
     * @return string
     */
    public function getStoreLabelSrc($storeLabel)
    {
        if ($storeLabel) {
            $storeLabel = \Our\Common::getStaticFile($storeLabel, \Our\ImageConst::storeLabel, 'ossHost');
        } else {
            $storeLabel = \Our\Common::getStaticFile(ImageConst::defaultStoreLabelName, ImageConst::defaultPath);
        }
        return $storeLabel;
    }
    /**
     * 店铺首页banner
     *
     * @param $storeLabel
     * @return string
     */
    public function getStoreBanner($img)
    {
        if ($img) {
            $storeLabel = \Our\Common::getStaticFile($img, \Our\ImageConst::storeLabel, 'ossHost');
        } else {
            $storeLabel = \Our\Common::getStaticFile(ImageConst::defaultStoreLabelName, ImageConst::defaultPath);
        }
        return $storeLabel;
    }
    /**
     * 店铺月销量
     *
     * @param $storeId
     * @return mixed
     */
    public function getStoreMonthSales($storeId)
    {
        $this->setDb();
        $data = $this->db->from('han_store_month_sales')->select('*')->where(array('store_id' => $storeId))->fetchOne();

        if ($data) {
            return $data['num'];
        }
    }


    /**
     * @param $field 可获取
     * @return mixed 返回店铺列表
     */
    public function getOnlineStores($where = null, $field = '*', $limit=[])
    {

        $conditionSql = 'store_state =1 and open_flag =1 and (store_end_time >=' . TIMESTAMP . ' or store_end_time =0 ) ';
        if ($where) {
            $conditionSql .= $where;
        }
        $result = $this->getStores($conditionSql, $field, $limit);
        return $result;
    }


    public function getList($where, $pageIndex, $pageSize, $field = '*')
    {
        $this->setDb();
        return $this->db->select($field)->from($this->_tableName)->where($where)->page($pageIndex, $pageSize)->order('state', 'ASC')->order('addtime', 'DESC')->fetchAll();
    }

    //是否收取快递费
    public function isChargeGet($storeId)
    {
        $isChargeGet = $this->getInfoById($storeId, 'is_charge_get');
        return $isChargeGet > ApiConst::zero ? TRUE : FALSE;
    }

    //线上支付订单已接单待发货退款是否收取服务费(0:否 1:是)
    public function isChargeNoSend($storeId)
    {
        $isChargeNoSend = $this->getInfoById($storeId, 'is_charge_no_send');
        return $isChargeNoSend;
    }

    //线上支付发货中订单退货是否收取服务费(0:否 1:是)
    public function isChargeSend($storeId)
    {
        $isChargeSend = $this->getInfoById($storeId, 'is_charge_send');
        return $isChargeSend;
    }

    public function isGetFee($storeId, $orderStatus)
    {
        if ($orderStatus <= ApiConst::orderStateWaitSend) {
            return $this->isChargeNoSend($storeId);
        }
        if ($orderStatus >= ApiConst::orderStateWaitRecieve) {
            return $this->isChargeGet($storeId);
        }
    }
    //获取是否应该收取服务费或快递费
    public function isFee($storeId,$shippingtype,$orderStatus)
    {
            if($shippingtype!=ApiConst::express){
                return $this->isGetFee($storeId,$orderStatus);
            }else{
                return $this->isChargeGet($storeId);
            }

    }

    /**
     * 获取店铺列表
     * @param $where
     * @param $field
     * @return mixed
     */
    public function getStores($where, $field = "*", $limit =[])
    {
        $this->setDb();
        if($limit){
            $result = $this->db->select($field)->from($this->_tableName)->where($where)->limit($limit[0],$limit[1])->fetchAll();
        }else{
            $result = $this->db->select($field)->from($this->_tableName)->where($where)->fetchAll();
        }
        return $result;
    }

    public function getStoresByCityCode($cityCode, $field = 'store_id,store_longitude,store_latitude,max_sign_rang,max_sales_rang,store_sales_scope')
    {
        $storeClassCondition = " and store_citycode = '" . $cityCode . "' ";

        $stores = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'), array($storeClassCondition, $field), \Our\ApiConst::oneHour, array($cityCode));
        return $stores;
    }

    public function delStoresCacheByCityCode($cityCode){
        \Our\RedisHelper::delCachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'),array(),array($cityCode));
    }

    /**
     * 获取当前城市对应正在经营店铺列表
     * @param $cityCode 当前城市编码
     * @return mixed 当前城市列表
     * @throws \Our\Exception
     */
    public function getExpressStores( $field = 'store_id,store_longitude,store_latitude,max_sign_rang,max_sales_rang,store_sales_scope')
    {

        $storeClassCondition = " and express_distribution = 1 ";

        $stores = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'), array($storeClassCondition, $field), \Our\ApiConst::oneHour);
        return $stores;
    }

    public function  deleteExpressStores($field = 'store_id,store_longitude,store_latitude,max_sign_rang,max_sales_rang,store_sales_scope'){
        $storeClassCondition = " and express_distribution = 1 ";
        \Our\RedisHelper::delCachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'),array(array($storeClassCondition, $field)));
    }

    public function getNearbyStores($cityCode, $storeIds)
    {
        $condition = " and store_citycode = '" . $cityCode . "' and store_id in (" . implode(',', $storeIds) . ")";
        $field = 'store_id as storeId,store_label as storeLabel,store_name as storeName,store_intro as storeIntro,store_longitude,store_latitude';
        $returnStores = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'), array($condition, $field), \Our\ApiConst::oneHour, array($cityCode));
        return $returnStores;
    }

    public function delNearbyStoresCache($cityCode){
        \Our\RedisHelper::delCachedFunction(\Redis\Db6\StoreRedisModel::getInstance(), array(&$this, 'getOnlineStores'),array(),array($cityCode));
    }

    public function getAddress($storeId){

        $storeInfo = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreRedisModel::getInstance(),array(&$this, 'getInfoById'),array($storeId),\Our\ApiConst::sevenDaySecond,array($storeId));
        //地址
        $address = '';
        if($storeInfo['area_info']){
            $address .= $storeInfo['area_info'];
        }
        if($storeInfo['store_address']){
            $address .= $storeInfo['store_address'];
        }
        return $address;

    }
    public function get($storeId,$label = true)
    {
        $storeInfo = \Our\RedisHelper::cachedFunction(\Redis\Db6\StoreRedisModel::getInstance(),array(&$this, 'getInfoById'),array($storeId),\Our\ApiConst::sevenDaySecond,array($storeId));
        if(!$storeInfo){
            \Error\ErrorModel::throwException(\Error\CodeConfigModel::storeNotExistOrClose);
        }

        if($label){
            //店铺头像
            $storeLabel = $this->getStoreLabelSrc($storeInfo['store_label']);

            return array($storeInfo,$storeLabel);
        }
        return $storeInfo;
    }
    /**
     * 店铺分享图片
     *
     * @param $storeLabel
     * @return string
     */
    public function getStoreShareSrc($img)
    {
        return \Our\Common::getStaticFile($img, \Our\ImageConst::defaultPath, 'ossHost');

    }

    /**
     * 获取店铺分享信息
     * @param $storeInfo
     * @return mixed
     */
    public function getShareInfo($storeInfo)
    {
        if(!is_array($storeInfo)){
            $storeInfo = $this->get($storeInfo,false);
        }
        $share['title'] = $storeInfo['store_share_title']?$storeInfo['store_share_title']:$storeInfo['store_name'];
        $share['desc']= $storeInfo['store_share_desc']?$storeInfo['store_share_desc']:$storeInfo['store_notice'];
        if($storeInfo['store_share_img']){
            $img = $this->getStoreShareSrc($storeInfo['store_share_img']);
        }else{
            $img = $this->getStoreLabelSrc($storeInfo['store_label']);
        }
        $share['imgUrl'] = $img;
        $share['link']='http://www.shenbd.com';
        return $share;
    }

    /**
     * 检查当前地址是否有可用配送方式
     * @param $address
     * @param $store
     * @return int
     */
    public function checkAddressInServiceArea($address,$store){
        $inAreaFlag = \Our\ApiConst::zero;
        if($store['buyer_distribution']||$store['express_distribution']){//如果店铺开启了到店自提和快递配送,则认为存在可用配送方式
            $inAreaFlag = \Our\ApiConst::one;
            return $inAreaFlag;
        }
        if($store['seller_distribution']&&$address){
            $str = str_replace('\"','"',$store['store_sales_scope']);
            $store_sales_scope = json_decode($str,true);
            $result = \Our\CommonExtension::isPointInPolygon($store_sales_scope, array('lng'=>$address['lng'],'lat'=>$address['lat']));
            if($result){
                $inAreaFlag = \Our\ApiConst::one;
            }
        }
        return $inAreaFlag;
    }


    /**
     * 类实例
     *
     * @var \DAO\UserModel
     */
    private static $_instance = null;

    /**
     * 获取类实例
     *
     * @return \DAO\UserModel
     */
    public static function getInstance($dbName=\Our\DbNameConst::salveDBConnectName)
    {
        if (!(self::$_instance instanceof self)) {
            self::$_instance = new self($dbName);
        }

        return self::$_instance;
    }

    private function _getColumns()
    {
        return array(
            'store_id',
            'store_name',
            'store_address',
            'store_state',
            'store_start_time',
            'store_close_time',
            'store_close_info',
            'store_label',
            'store_banner',
            'store_phone',
            'store_zy',
            'store_credit',
            'member_id',
            'store_desccredit',
            'store_servicecredit',
            'store_deliverycredit',
            'store_collect',
            'store_slide',
            'store_slide_url',
            'store_workingtime',
            'store_free_price',
            'store_decoration_switch',
            'store_decoration_only',
            'store_free_time',
            'store_longitude',
            'store_latitude',
            'wx_pay',
            'offline_pay',
            'aog_pay',
            'seller_distribution',
            'start_shipping_price',
            'free_shipping_price',
            'buyer_distribution',
            'express_distribution',
            'is_ally',
            'is_ally_discount',
            'ally_discount',
            'recommend_word',
            'is_direct_refund',
            'is_charge_no_send',
            'is_charge_send',
            'is_charge_get',
            'is_charge_delivery',
            'search_keyword',
            'store_sales_scope',
            'max_sales_rang',
            'store_notice',
            'store_citycode',
            'store_subaddress',
            'store_qrcode_param',
            'store_qrcode_path',
            'store_detail_image',
            'store_dis_sta_time',
            'store_dis_end_time',
            'store_share_title',
            'store_share_img',
            'store_share_desc',
            'store_share_url',
            'is_wx_bind',
            'default_user_comment',
            'store_end_time',
            'store_points_ratio',
            'delivery_interval',
            'is_rec',
            'offline_payway',
            'open_flag',
            'store_intro',
            'business_start_time',
            'business_end_time',
        );
    }

}
